/* secure 8BitAmEthernet protocol client
 * version 0.1 date 2005-03-07
 * Copyright 2005 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 * a project of CCC-AC - http://www.cccac.de/
 */

/* secure 8BitAmEthernet protocol (S8P)
 * 
 * based on symmetric XTEA cipher (ENC_k, DEC_k) with 64 rounds and 128 bit key (k)
 *
 * protocol:
 *   client:
 *     connects to server (TCP port 8)
 *   server: 
 *     chose 8 byte of random data: d
 *     send challenge: c = ENC_k( ("BitAmEth" XOR d) . d )
 *   client:
 *     receive c and decrypt it: x . d2 = DEC_k( c )
 *     test if x == "BitAmEth" XOR d2
 *     close connection if not
 *     let o be the 8 byte command to be sent
 *     send response: r = ENC_k( "Client  " XOR d2) . o )
 *   server:
 *     receive r and decrypt it: y . o2 = DEC_k( r )
 *     test if y == "Client  " XOR d
 *     close connection if not
 *     execute command o2
 *     let s be the 8 byte status returned by this command
 *     send acknowledge: a = ENC_k( ("Status  " XOR d) . s )
 *   client:
 *     receive a and decrypt it: z . a2 = DEC_k( q )
 *     test if z == "Status  " XOR d2
 *     close connection if not
 *     interpret acknowledge a2
 *     close connection
 *   server is 8BitAmEthernet
 *   "." is the appending operator
 *
 *   commands are identified by their first byte
 *   the status has alwys got the same first byte as the command
 *
 *   get output
 *     command: 0x00 7x<unused>
 *       unused: 0x00
 *     status: 0x00 <state> 6x<unused>
 *       state: bit 0 corresponds to output 1, ..., bit 7 corresponds to output 8
 *       unused: 0x00
 *
 *   change output
 *     command: 0x01 <action> <output> 5x<unused>
 *       action: 0x00=turn off, 0x01=turn on, 0xFF=toggle
 *       output: number of output port (1..8)
 *       unused: 0x00
 *     status: 0x01 <state> 6x<unused>
 *       state: bit 0 corresponds to output 1, ..., bit 7 corresponds to output 8
 *       unused: 0x00
 */

#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "s8p.h"
#include "xtea.h"

//how many XTEA rounds to use
#define S8P_XTEA_ROUNDS 64

//execute S8P command
//returns 0 on success, -1 on error
int S8pCommand( in_addr_t Ip, in_port_t Port, unsigned char Key[16], unsigned char Command[8], unsigned char Status[8] )
{
  int sock, i, pos;
  struct sockaddr_in addr;
  uint64_t xteaCtx;
  unsigned char random[8], challenge[16], response[16], acknowledge[16];

  //establish connection
  sock = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
  if( sock == -1 )
    return -1;
  addr.sin_family = AF_INET;
  addr.sin_port = htons( Port );
  addr.sin_addr.s_addr = htonl( Ip );
  if( connect( sock, (struct sockaddr *)&addr, sizeof( addr ) ) != 0 )
  {
    close( sock );
    return -1;
  }

  //receive challenge c
  for( pos = 0; pos < (int)sizeof( challenge ); pos += i )
  {
    i = recv( sock, challenge + pos, sizeof( challenge ) - pos, 0 );
    if( i <= 0 )
    {
      if( i == 0 )
        errno = EPROTO;
      close( sock );
      return -1;
    }
  }
  //decrypt challenge c: x . d2 = DEC_k( c )
  xteaCtx = 0;
  xtea_dec( (uint32_t *)Key, S8P_XTEA_ROUNDS,
            (uint64_t *)challenge, (uint64_t *)challenge,
            sizeof( challenge ) / sizeof( uint64_t ),
            &xteaCtx );
  for( i = 0; i < (int)sizeof( random ); i++ )
    random[i] = challenge[i + sizeof( random )];
  //test if x == "BitAmEth" XOR d2
  for( i = 0; i < (int)sizeof( random ); i++ )
    if( challenge[i] != (((unsigned char * )"BitAmEth")[i] ^ random[i]) )
      break;
  //close connection if not
  if( i < (int)sizeof( random ) )
  {
    errno = EPROTO;
    close( sock );
    return -1;
  }
  //let o be the 8 byte command to be sent
  //assemble response: r = ENC_k( "Client  " XOR d2) . o )
  for( i = 0; i < (int)sizeof( random ); i++ )
  {
    response[i] = ((unsigned char *)"Client  ")[i] ^ random[i];
    response[i + sizeof( random )] = Command[i];
  }
  xteaCtx = 0;
  xtea_enc( (uint32_t *)Key, S8P_XTEA_ROUNDS,
            (uint64_t *)response, (uint64_t *)response,
            sizeof( response ) / sizeof( uint64_t ),
            &xteaCtx );
  //send response r
  for( pos = 0; pos < (int)sizeof( response ); pos += i )
  {
    i = send( sock, response + pos, sizeof( response ) - pos, 0 );
    if( i <= 0 )
    {
      if( i == 0 )
        errno = EPROTO;
      close( sock );
      return -1;
    }
  }

  //receive acknowledge a
  for( pos = 0; pos < (int)sizeof( acknowledge ); pos += i )
  {
    i = recv( sock, acknowledge + pos, sizeof( acknowledge ) - pos, 0 );
    if( i <= 0 )
    {
      if( i == 0 )
        errno = EPROTO;
      close( sock );
      return -1;
    }
  }
  //decrypt acknowledge a: z . a2 = DEC_k( q )
  xteaCtx = 0;
  xtea_dec( (uint32_t *)Key, S8P_XTEA_ROUNDS,
            (uint64_t *)acknowledge, (uint64_t *)acknowledge,
            sizeof( acknowledge ) / sizeof( uint64_t ),
            &xteaCtx );
  //test if z == "Status  " XOR d2
  for( i = 0; i < (int)sizeof( random ); i++ )
    if( acknowledge[i] != (((unsigned char * )"Status  ")[i] ^ random[i]) )
      break;
  //close connection if not
  if( i < (int)sizeof( random ) )
  {
    errno = EPROTO;
    close( sock );
    return -1;
  }
  //interpret acknowledge a2
  for( i = 0; i < (int)sizeof( random ); i++ )
    Status[i] = acknowledge[i + sizeof( random )];
  //close connection
  close( sock );

  return 0;  
}

//execute S8P get output
//returns 0 on success, -1 on error
int S8pGetOutput( in_addr_t Ip, in_port_t Port, unsigned char Key[16], unsigned char * pState )
{
  unsigned char command[8], status[8];
  int retval;
 
  //command: 0x00 7x<unused>
  command[0] = 0x00;
  memset( command+1, 0x00, 7 );

  //execute command
  retval = S8pCommand( Ip, Port, Key, command, status );
  if( retval < 0 )
    return -1;

  //status: 0x00 <state> 6x<unused>
  if( status[0] != 0x00 )
  {
    errno = EBADMSG;
    return -1;
  }
  *pState = status[1];

  return 0;
}

//execute S8P change output
//Action: 1=turn on, 0=turn off, -1=toggle
//Output: 1..8
//returns 0 on success, -1 on error
int S8pChangeOutput( in_addr_t Ip, in_port_t Port, unsigned char Key[16], signed char Action, unsigned char Output, unsigned char * pState )
{
  unsigned char command[8], status[8];
  int retval;

  //check parameters
  if( Output < 1 || Output > 8 )
  {
    errno = EINVAL;
    return -1;
  }

  //command: 0x01 <action> 6x<unused>
  command[0] = 0x01;
  command[1] = Action > 0 ? 0x01 : Action == 0 ? 0x00 : 0xFF;
  command[2] = Output;
  memset( command+3, 0x00, 5 );

  //execute command
  retval = S8pCommand( Ip, Port, Key, command, status );
  if( retval < 0 )
    return -1;

  //status: 0x01 <state> 6x<unused>
  if( status[0] != 0x01 )
  {
    errno = EBADMSG;
    return -1;
  }
  *pState = status[1];

  return 0;
}

