/* secure 8BitAmEthernet protocol client
 * version 0.1 date 2005-03-07
 * Copyright 2005 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 * a project of CCC-AC - http://www.cccac.de/
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "s8p.h"


#define count( array ) (sizeof( (array) ) / sizeof( (array[0]) ))


//the ip and port to connect to
in_addr_t Ip = INADDR_LOOPBACK;
in_port_t Port = 8;

//the key
unsigned char Key[16] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };


//set the symmetric key to use for encryption
int cmd_key( int arg_c, char * * arg_v )
{
  char * p_key;
  int i, j, invalid, truncated;
  unsigned char val;

  //get parameters
  if( arg_c < 2 )
  {
    fprintf( stderr, "missing key data for command \"key\"\n\n" );
    return -1;
  }
  p_key = arg_v[1];

  //convert key data to key
  i = 0;
  j = 0;
  val = 0;
  invalid = 0;
  truncated = 0;
  for( ; *p_key != 0; p_key++ )
  {
    if( *p_key >= '0' && *p_key <= '9' )
    {
      val = val * 10 + *p_key - '0';
      j++;
      if( j >= 3 )
      {
        if( i < (int)count( Key ) )
        {
          Key[i] = val;
          i++;
        }
        else
          truncated = 1;
        j = 0;
        val = 0;
      }
    }
    else
      invalid = 1;
  }

  //show warnings
  if( invalid )
    fprintf( stderr, "invalid characters in key data were ignored\n" );
  if( truncated )
    fprintf( stderr, "trailing characters in key data were truncated\n" );
  if( j > 0 )
    fprintf( stderr, "incomplete byte at end of key data was ignored\n" );
  if( i < (int)count( Key ) )
    fprintf( stderr, "key data too short, padded with 0\n" );

  //fill rest of key with 0
  for( ; i < (int)count( Key ); i++ )
    Key[i] = 0;

  //2 parameters processed
  return 2;
}


//show the command to enter at 8BitAmEthernet to set the key
int cmd_keycmd( int arg_c, char * * arg_v )
{
  int i;

  //show 8BitAmEthernet command
  printf( "command to set key: * 2 1" );
  for( i = 0; i < (int)count( Key ); i++ )
    printf( " %03d", Key[i] );
  printf( " #\n" );
  
  //1 parameter processed
  return 1;

  //keep compiler happy
  arg_c = 0;
  arg_v = NULL;
}


//set IP and port to connect to
int cmd_connect( int arg_c, char * * arg_v )
{
  char * p_ip, * p_port, * p_chr;
  unsigned long l;

  //get parameters
  if( arg_c < 2 )
  {
    fprintf( stderr, "missing IP for command \"connect\"\n\n" );
    return -1;
  }
  p_ip = arg_v[1];
  if( arg_c < 3 )
  {
    fprintf( stderr, "missing port for command \"connect\"\n\n" );
    return -1;
  }
  p_port = arg_v[2];

  //parse IP
  Ip = ntohl( inet_addr( p_ip ) );
  if( Ip == INADDR_NONE )
  {
    fprintf( stderr, "cannot parse IP \"%s\"\n\n", p_ip );
    return -1;
  }

  //parse port
  l = strtoul( p_port, &p_chr, 0 );
  if( *p_port == 0 || *p_chr != 0 || l < 1 || l > 65535 )
  {
    fprintf( stderr, "cannot parse port \"%s\"\n\n", p_port );
    return -1;
  }
  Port = (in_port_t)l;

  //3 parameters processed
  return 3;
}


//execute raw command
int cmd_raw( int arg_c, char * * arg_v )
{
  char * p_raw;
  int i, j, invalid, truncated;
  unsigned char val, raw[8], status[8];

  //get parameters
  if( arg_c < 2 )
  {
    fprintf( stderr, "missing raw command data for command \"raw\"\n\n" );
    return -1;
  }
  p_raw = arg_v[1];

  //convert raw command to binary
  i = 0;
  j = 0;
  val = 0;
  invalid = 0;
  truncated = 0;
  for( ; *p_raw != 0; p_raw++ )
  {
    if( (*p_raw >= '0' && *p_raw <= '9') ||
        (*p_raw >= 'A' && *p_raw <= 'F') ||
        (*p_raw >= 'a' && *p_raw <= 'f') )
    {
      if( *p_raw >= '0' && *p_raw <= '9' )
        val = val * 0x10 + *p_raw - '0';
      else if( *p_raw >= 'A' && *p_raw <= 'Z' )
        val = val * 0x10 + *p_raw - 'A' + 0xA;
      else if( *p_raw >= 'a' && *p_raw <= 'z' )
        val = val * 0x10 + *p_raw - 'a' + 0xA;
      j++;
      if( j >= 2 )
      {
        if( i < (int)count( raw ) )
        {
          raw[i] = val;
          i++;
        }
        else
          truncated = 1;
        j = 0;
        val = 0;
      }
    }
    else
      invalid = 1;
  }

  //show warnings
  if( invalid )
    fprintf( stderr, "invalid characters in raw command data were ignored\n" );
  if( truncated )
    fprintf( stderr, "trailing characters in raw command data were truncated\n" );
  if( j > 0 )
    fprintf( stderr, "incomplete byte at end of raw command data was ignored\n" );
  if( i < (int)count( raw ) )
    fprintf( stderr, "raw command data too short, padded with 0\n" );

  //fill rest of raw command with 0
  for( ; i < (int)count( raw ); i++ )
    raw[i] = 0;

  //show raw command
  printf( "executing raw command" );
  for( i = 0; i < (int)count( raw ); i++ )
    printf( " %02X", raw[i] );
  printf( "...\n" );

  //execute command
  if( S8pCommand( Ip, Port, Key, raw, status ) != 0 )
  {
    fprintf( stderr, "raw command failed: %s\n\n", strerror( errno ) );
    return -1;
  }

  //show status
  printf( "command returned status" );
  for( i = 0; i < (int)count( status ); i++ )
    printf( " %02X", status[i] );
  printf( "\n" );

  //2 parameters processed
  return 2;
}


//get output state
int cmd_getout( int arg_c, char * * arg_v )
{
  unsigned char state;
  int i;

  //get output state
  if( S8pGetOutput( Ip, Port, Key, &state ) != 0 )
  {
    fprintf( stderr, "get output state failed: %s\n\n", strerror( errno ) );
    return -1;
  }

  //show output state
  printf( "current output state:" );
  for( i = 0; i < 8; i++, state >>= 1 )
    printf( " %s", state & 0x01 ? "on" : "off" );
  printf( "\n" );
  
  //1 parameter processed
  return 1;

  //keep compiler happy
  arg_c = 0;
  arg_v = NULL;
}


//change output state
int cmd_out( int arg_c, char * * arg_v )
{
  char * p_action, * p_output, * p_chr;
  signed char action;
  unsigned char output, state;
  unsigned long l;
  int i;

  //get parameters
  if( arg_c < 2 )
  {
    fprintf( stderr, "missing action for command \"out\"\n\n" );
    return -1;
  }
  p_action = arg_v[1];
  if( arg_c < 3 )
  {
    fprintf( stderr, "missing output port for command \"out\"\n\n" );
    return -1;
  }
  p_output = arg_v[2];

  //parse action
  if( strcasecmp( p_action, "on") == 0 )
    action = 0x01;
  else if( strcasecmp( p_action, "off") == 0 )
    action = 0x00;
  else if( strcasecmp( p_action, "toggle") == 0 )
    action = 0xFF;
  else
  {
    fprintf( stderr, "cannot parse action \"%s\"\n\n", p_action );
    return -1;
  }

  //parse output port
  l = strtoul( p_output, &p_chr, 0 );
  if( *p_output == 0 || *p_chr != 0 || l < 1 || l > 8 )
  {
    fprintf( stderr, "cannot parse output port \"%s\"\n\n", p_output );
    return -1;
  }
  output = (unsigned char)l;

  //change output state
  if( S8pChangeOutput( Ip, Port, Key, action, output, &state ) != 0 )
  {
    fprintf( stderr, "changing output state failed: %s\n\n", strerror( errno ) );
    return -1;
  }

  //show output state
  printf( "new output state:" );
  for( i = 0; i < 8; i++, state >>= 1 )
    printf( " %s", state & 0x01 ? "on" : "off" );
  printf( "\n" );
  
  //3 parameters processed
  return 3;
}


//table with known commands
struct t_cmd_tab
{
  char * name;
  int (*func)( int arg_c, char * * arg_v );
} cmd_tab[] =
{
  { "key", cmd_key },
  { "keycmd", cmd_keycmd },
  { "connect", cmd_connect },
  { "raw", cmd_raw },
  { "getout", cmd_getout },
  { "out", cmd_out },
};


//main program
int main( int arg_c, char * * arg_v )
{
  int i, j, retval;

  //print short info
  printf( "secure 8BitAmEthernet protocol client\n"
          "version 0.1 date 2005-03-07\n"
          "Copyright 2005 Stefan Schuermans <1stein@schuermans.info>\n"
          "Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html\n"
          "a project of CCC-AC - http://www.cccac.de/\n\n" );

  //display help if no parameters
  if( arg_c < 2 )
  {
    printf( "syntax: s8pc [command] [...]\n\n"
            "commands: key <byte 0 (3 decimal digits)>[...]<byte 15 (3 decimal digits)>\n"
            "            set the symmetric key to use for encryption\n"
            "          keycmd\n"
            "            show the command to enter at 8BitAmEthernet to set the key\n"
            "          connect <ip> <port>\n"
            "            set IP and port to connec to\n"
            "          raw <byte 0 (2 hexadecimal digits)>[...]<byte 7 (2 hexadecimal digits)>\n"
            "            execute raw command\n"
            "          getout\n"
            "            get output state\n"
            "          out [on|off|toggle] <output port>\n"
            "            change output state\n"
            "examples: s8pc key 023[...]042 keycmd\n"
            "          s8pc key 023[...]042 connect 192.168.0.8:8 raw 00[...]00\n\n" );
    return 1;
  }

  //process commands
  for( i = 1; i < arg_c; )
  {
    //search command in list
    for( j = 0; j < (int)count( cmd_tab ); j++ )
      if( strcasecmp( arg_v[i], cmd_tab[j].name ) == 0 )
        break;
    //found command
    if( j < (int)count( cmd_tab ) )
    {
      retval = cmd_tab[j].func( arg_c - i, arg_v + i );
      if( retval <= 0 )
        return -1;
      i += retval;
    }
    //unknown command
    else
    {
      fprintf( stderr, "unknown command \"%s\"\n\n", arg_v[i] );
      return -1;
    }
  } //for( i ...

  printf( "\n" );
  return 0;
}
