/* 8BitAmEthernet
 * version 0.4.0 date 2006-04-13
 * Copyright 2005-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 * a project of CCC-AC - http://www.cccac.de/
 */

/* secure 8BitAmEthernet protocol (S8P)
 * 
 * based on symmetric XTEA cipher (ENC_k, DEC_k) with 64 rounds and 128 bit key (k)
 *
 * protocol:
 *   client:
 *     connects to server (TCP port 8)
 *   server: 
 *     chose 8 byte of random data: d
 *     send challenge: c = ENC_k( ("BitAmEth" XOR d) . d )
 *   client:
 *     receive c and decrypt it: x . d2 = DEC_k( c )
 *     test if x == "BitAmEth" XOR d2
 *     close connection if not
 *     let o be the 8 byte command to be sent
 *     send response: r = ENC_k( "Client  " XOR d2) . o )
 *   server:
 *     receive r and decrypt it: y . o2 = DEC_k( r )
 *     test if y == "Client  " XOR d
 *     close connection if not
 *     execute command o2
 *     let s be the 8 byte status returned by this command
 *     send acknowledge: a = ENC_k( ("Status  " XOR d) . s )
 *   client:
 *     receive a and decrypt it: z . a2 = DEC_k( q )
 *     test if z == "Status  " XOR d2
 *     close connection if not
 *     interpret acknowledge a2
 *     close connection
 *   server is 8BitAmEthernet
 *   "." is the appending operator
 *
 *   commands are identified by their first byte
 *   the status has always got the same first byte as the command
 *
 *   get output
 *     command: 0x00 7x<unused>
 *       unused: 0x00
 *     status: 0x00 <state> 6x<unused>
 *       state: bit 0 corresponds to output 1, ..., bit 7 corresponds to output 8
 *       unused: 0x00
 *
 *   change output
 *     command: 0x01 <action> <output> 5x<unused>
 *       action: 0x00=turn off, 0x01=turn on, 0xFF=toggle
 *       output: number of output port (1..8)
 *       unused: 0x00
 *     status: 0x01 <state> 6x<unused>
 *       state: bit 0 corresponds to output 1, ..., bit 7 corresponds to output 8
 *       unused: 0x00
 */

#include <string.h>
#include <avr/io.h>

#include "config.h"
#include "debug.h"
#include "macros.h"
#include "output.h"
#include "random.h"
#include "s8p.h"
#include "tcp.h"
#include "xtea.h"

//how many XTEA rounds to use
#define S8P_XTEA_ROUNDS 64

//states of S8P connections
#define S8P_FREE 0 //free entry
#define S8P_NEW 1 //new connection, waiting for challenge to be generated
#define S8P_CHALLENGE 2 //sending challenge
#define S8P_RESPONSE 3 //receiving response
#define S8P_COMMAND 4 //waiting for execution of command
#define S8P_ACKNOWLEDGE 5 //sending acknowledge
#define S8P_CLOSE 6 //close connection

//table with S8P connections
struct S8pConnection
{
  unsigned char State; //current state
  unsigned char ConnNo; //number of the TCP connection
  unsigned char Random[8]; //random data selected for this connection
  unsigned char Buffer[16]; //the challenge to be sent by 8BitAmEthernet (S8P_CHALLENGE)
                            //the response sent by the client (S8P_RESPONSE)
                            //the acknowledge to be sent by 8BitAmEthernet (S8P_ACKNOWLEDGE)
} S8pConns[4];

//called when connection is established
void S8pConnect( unsigned char ConnNo )
{
  unsigned char i;
  struct S8pConnection * pConn;

  //find connection in table (in case TCP calls us twice, this should never happen)
  for( i = 0; i < count( S8pConns ); i++ )
    if( S8pConns[i].State != S8P_FREE &&
        S8pConns[i].ConnNo == ConnNo )
      break;
  //connection found
  if( i < count( S8pConns ) )
  {
    //free this entry
    S8pConns[i].State = S8P_FREE;
    return;
  }

  //find a free entry
  for( i = 0; i < count( S8pConns ); i++ )
    if( S8pConns[i].State == S8P_FREE )
      break;
  if( i >= count( S8pConns ) ) //no free entry found
    return; //ignore this connection (will be closed in first call of S8pSend)
  //get pointer to connection
  pConn = &S8pConns[i];

  debug_s8p_printf( "accept no=%u", ConnNo );

  //put new connection into table
  pConn->State = S8P_NEW;
  pConn->ConnNo = ConnNo;
}

//called when connection is closed / reset
//(after this, the connection number may not be used any more)
void S8pClose( unsigned char ConnNo )
{
  unsigned char i;
  struct S8pConnection * pConn;

  //find connection in table
  for( i = 0; i < count( S8pConns ); i++ )
    if( S8pConns[i].State != S8P_FREE &&
        S8pConns[i].ConnNo == ConnNo )
      break;
  if( i >= count( S8pConns ) ) //connection not found
    return; //ignore this (now already closed) connection
  //get pointer to connection
  pConn = &S8pConns[i];

  debug_s8p_printf( "close no=%u", ConnNo );

  //drop connection from table
  pConn->State = S8P_FREE;
}

//called when sending data is possible
//(return length of available data, 0xFFFF to close connection)
unsigned short S8pSend( unsigned char ConnNo, unsigned long Pos, unsigned char * pBuffer, unsigned short MaxLen )
{
  unsigned char i;
  struct S8pConnection * pConn;
  unsigned short len;

  //find connection in table
  for( i = 0; i < count( S8pConns ); i++ )
    if( S8pConns[i].State != S8P_FREE &&
        S8pConns[i].ConnNo == ConnNo )
      break;
  if( i >= count( S8pConns ) ) //connection not found
    return 0xFFFF; //close connection
  //get pointer to connection
  pConn = &S8pConns[i];

  //different states
  switch( pConn->State )
  {

    case S8P_NEW:
    case S8P_RESPONSE:
    case S8P_COMMAND:
      debug_s8p_printf( "send no=%u nothing", ConnNo );
      return 0; //send nothing

    case S8P_CHALLENGE: //sending challenge
      if( Pos > sizeof( pConn->Buffer ) ) //behind challenge
        return 0; //send nothing
      len = min( sizeof( pConn->Buffer ) - Pos, MaxLen ); //send (part of) challenge
      memcpy( pBuffer, pConn->Buffer + Pos, len );
      debug_s8p_printf( "send no=%u challenge pos=%lu len=%u", ConnNo, Pos, len );
      return len;

    case S8P_ACKNOWLEDGE: //sending acknowledge
      if( Pos < sizeof( pConn->Buffer ) ) //before acknowledge
        return 0xFFFF; //this cannot happen - internal error - close connection
      if( Pos > sizeof( pConn->Buffer ) * 2 ) //behind acknowledge
        return 0; //send nothing
      len = min( sizeof( pConn->Buffer ) * 2 - Pos, MaxLen ); //send (part of) acknowledge
      memcpy( pBuffer, pConn->Buffer + Pos - sizeof( pConn->Buffer ), len );
      debug_s8p_printf( "send no=%u acknowledge pos=%lu len=%u", ConnNo, Pos - sizeof( pConn->Buffer ), len );
      return len;

    case S8P_CLOSE:
    default:
      return 0xFFFF; //close connection

  } //switch( pConn->State )
}

//called when data was sent and ACKed
void S8pSent( unsigned char ConnNo, unsigned long Pos )
{
  unsigned char i;
  struct S8pConnection * pConn;

  //find connection in table
  for( i = 0; i < count( S8pConns ); i++ )
    if( S8pConns[i].State != S8P_FREE &&
        S8pConns[i].ConnNo == ConnNo )
      break;
  if( i >= count( S8pConns ) ) //connection not found
    return; //ignore
  //get pointer to connection
  pConn = &S8pConns[i];

  //different states
  switch( pConn->State )
  {

    case S8P_CHALLENGE: //sending challenge
      if( Pos < sizeof( pConn->Buffer ) ) //entire challenge not yet sent and ACKed
        break;
      //entire challenge is now sent and ACKed
      pConn->State = S8P_RESPONSE; //receive response now
      break;

    case S8P_ACKNOWLEDGE: //sending acknowledge
      if( Pos < sizeof( pConn->Buffer ) * 2 ) //entire acknowledge not yet sent and ACKed
        break;
      //entire acknowledge is now sent and ACKed
      pConn->State = S8P_CLOSE; //close connection now
      break;

  } //switch( pConn->State )
}

//called when data was received
//must return new window size (not smaller than curWnd)
unsigned short S8pReceived( unsigned char ConnNo, unsigned long Pos, unsigned char * pBuffer, unsigned short Len, unsigned short curWnd )
{
  unsigned char i;
  struct S8pConnection * pConn;

  //find connection in table
  for( i = 0; i < count( S8pConns ); i++ )
    if( S8pConns[i].State != S8P_FREE &&
        S8pConns[i].ConnNo == ConnNo )
      break;
  if( i >= count( S8pConns ) ) //connection not found
    return max( curWnd, 32 ); //ignore this connection (will be closed in first call of S8pSend)
  //get pointer to connection
  pConn = &S8pConns[i];

  //different states
  switch( pConn->State )
  {

    case S8P_RESPONSE: //receiving response
      debug_s8p_printf( "recv no=%u response pos=%lu len=%u", ConnNo, Pos, Len );
      if( Pos < sizeof( pConn->Buffer ) ) //not behind response
        memcpy( pConn->Buffer + Pos, pBuffer, min( Len, sizeof( pConn->Buffer ) - Pos ) );
      if( Pos + Len >= sizeof( pConn->Buffer ) ) //entire response is received
        pConn->State = S8P_COMMAND; //wait for execution of command
      break;

  } //switch( pConn->State )

  //return at least 32 bytes window size
  // - we are always able to receive data
  return max( curWnd, 32 );
}

//s8p notification functions
struct TcpNotify S8pNotify = //(extern)
{
  .Connect = S8pConnect,
  .Close = S8pClose,
  .Send = S8pSend,
  .Sent = S8pSent,
  .Received = S8pReceived,
};

//initialize
void S8pInit( void ) //(extern)
{
  unsigned char i;

  //no S8P connections yet
  for( i = 0; i < count( S8pConns ); i++ )
    S8pConns[i].State = S8P_FREE;
}

//execute a S8P command
static void S8pCommand( unsigned char Command[8], unsigned char Status[8] )
{
  //commands are identified by their first byte
  switch( Command[0] )
  {

    //get output
    case 0x00:
      //command: 0x00 7x<unused>
      //status: 0x00 <state> 6x<unused>
      Status[0] = 0x00;
      Status[1] = OutputGetState( );
      memset( Status + 2, 0x00, sizeof( Status ) - 2 );
      break;

    //change output
    case 0x01:
      //command: 0x01 <action> <output> 5x<unused>
      OutputChange( Command[1], Command[2] );
      //status: 0x01 <state> 6x<unused>
      Status[0] = 0x01;
      Status[1] = OutputGetState( );
      memset( Status + 2, 0x00, sizeof( Status ) - 2 );
      break;

    //unknown command
    default:
      memset( Status, 0xFF, sizeof( Status ) );

  } //switch( Command[0] );
}

//task function to do the work - call from main loop
void S8pTask( void ) //(extern)
{
  static unsigned char conn = -1; //used to start serching at another connection each time
  unsigned char i, req_send, Command[8];
  struct S8pConnection * pConn;
  uint64_t xteaCtx;

  //search next S8P connection that has work to be done
  for( i = 0; i < count( S8pConns ); i++ )
  {
    conn++; //next connection
    if( conn >= count( S8pConns ) )
      conn = 0;
    if( S8pConns[conn].State == S8P_NEW || //check for work to do
        S8pConns[conn].State == S8P_COMMAND )
      break;
  }
  if( i >= count( S8pConns ) ) //found none
    return;
  pConn = &S8pConns[i]; //get pointer to connection

  //different states
  req_send = 0;
  switch( pConn->State )
  {

    //new connection, waiting for challenge to be generated
    case S8P_NEW:
      //generate random data: d
      RandomGetData( pConn->Random, sizeof( pConn->Random ) );
      //generate challenge: c = ENC_k( ("BitAmEth" XOR d) . d )
      for( i = 0; i < sizeof( pConn->Random ); i++ )
        pConn->Buffer[i] = ((unsigned char *)"BitAmEth")[i] ^ pConn->Random[i];
      memcpy( pConn->Buffer + sizeof( pConn->Random ), pConn->Random, sizeof( pConn->Random ) );
      xteaCtx = 0;
      xtea_enc( (uint32_t *)ConfigS8pKey, S8P_XTEA_ROUNDS,
                (uint64_t *)pConn->Buffer, (uint64_t *)pConn->Buffer,
                sizeof( pConn->Buffer ) / sizeof( uint64_t ), &xteaCtx );
      //now send challenge
      pConn->State = S8P_CHALLENGE;
      req_send = 1;
      break;

    //waiting for execution of command
    case S8P_COMMAND:
      //decrypt response: y . o2 = DEC_k( r )
      xteaCtx = 0;
      xtea_dec( (uint32_t *)ConfigS8pKey, S8P_XTEA_ROUNDS,
                (uint64_t *)pConn->Buffer, (uint64_t *)pConn->Buffer,
                sizeof( pConn->Buffer ) / sizeof( uint64_t ), &xteaCtx );
      //test if y == "Client  " XOR d
      for( i = 0; i < sizeof( pConn->Random ); i++ )
        if( pConn->Buffer[i] != (((unsigned char *)"Client  ")[i] ^ pConn->Random[i]) )
          break;
      //close connection if not
      if( i < sizeof( pConn->Random ) )
      {
        pConn->State = S8P_CLOSE;
        req_send = 1;
        break;
      }
      //execute command o2, let s be the status returned by this command
      memcpy( Command, pConn->Buffer + sizeof( pConn->Random ), sizeof( Command ) );
      debug_s8p_printf( "command no=%u %02X %02X %02X %02X %02X %02X %02X %02X",
                        pConn->ConnNo,
                        pConn->Buffer[sizeof( pConn->Random ) + 0],
                        pConn->Buffer[sizeof( pConn->Random ) + 1],
                        pConn->Buffer[sizeof( pConn->Random ) + 2],
                        pConn->Buffer[sizeof( pConn->Random ) + 3],
                        pConn->Buffer[sizeof( pConn->Random ) + 4],
                        pConn->Buffer[sizeof( pConn->Random ) + 5],
                        pConn->Buffer[sizeof( pConn->Random ) + 6],
                        pConn->Buffer[sizeof( pConn->Random ) + 7] );
      S8pCommand( Command, pConn->Buffer + sizeof( pConn->Random ) );
      debug_s8p_printf( "status no=%u %02X %02X %02X %02X %02X %02X %02X %02X",
                        pConn->ConnNo,
                        pConn->Buffer[sizeof( pConn->Random ) + 0],
                        pConn->Buffer[sizeof( pConn->Random ) + 1],
                        pConn->Buffer[sizeof( pConn->Random ) + 2],
                        pConn->Buffer[sizeof( pConn->Random ) + 3],
                        pConn->Buffer[sizeof( pConn->Random ) + 4],
                        pConn->Buffer[sizeof( pConn->Random ) + 5],
                        pConn->Buffer[sizeof( pConn->Random ) + 6],
                        pConn->Buffer[sizeof( pConn->Random ) + 7] );
      //generate acknowledge: a = ENC_k( ("Status  " XOR d) . s )
      for( i = 0; i < sizeof( pConn->Random ); i++ )
        pConn->Buffer[i] = ((unsigned char *)"Status  ")[i] ^ pConn->Random[i];
      xteaCtx = 0;
      xtea_enc( (uint32_t *)ConfigS8pKey, S8P_XTEA_ROUNDS,
                (uint64_t *)pConn->Buffer, (uint64_t *)pConn->Buffer,
                sizeof( pConn->Buffer ) / sizeof( uint64_t ), &xteaCtx );
      //now send acknowledge
      pConn->State = S8P_ACKNOWLEDGE;
      req_send = 1;
      break;

  } //switch( pConn->State )

  //request to send next packet
  if( req_send )
    TcpSend( pConn->ConnNo );
}

