/* 8BitAmEthernet
 * version 0.4.0 date 2006-04-13
 * Copyright 2005-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 * a project of CCC-AC - http://www.cccac.de/
 */

#include <avr/io.h>

#include "code.h"
#include "debug.h"
#include "keypad.h"
#include "macros.h"
#include "output.h"
#include "random.h"
#include "timing.h"

//IO pins of keypad keys
#define KEYPAD_0_IN( ) (bit_clear( DDRB, PB3 ))
#define KEYPAD_0_PU( ) (bit_set( PORTB, PB3 ))
#define KEYPAD_0( ) (bit_is_clear( PINB, PB3 ))
#define KEYPAD_1_IN( ) (bit_clear( DDRB, PB4 ))
#define KEYPAD_1_PU( ) (bit_set( PORTB, PB4 ))
#define KEYPAD_1( ) (bit_is_clear( PINB, PB4 ))
#define KEYPAD_2_IN( ) (bit_clear( DDRE, PE7 ))
#define KEYPAD_2_PU( ) (bit_set( PORTE, PE7 ))
#define KEYPAD_2( ) (bit_is_clear( PINE, PE7 ))
#define KEYPAD_3_IN( ) (bit_clear( DDRE, PE2 ))
#define KEYPAD_3_PU( ) (bit_set( PORTE, PE2 ))
#define KEYPAD_3( ) (bit_is_clear( PINE, PE2 ))
#define KEYPAD_4_IN( ) (bit_clear( DDRB, PB5 ))
#define KEYPAD_4_PU( ) (bit_set( PORTB, PB5 ))
#define KEYPAD_4( ) (bit_is_clear( PINB, PB5 ))
#define KEYPAD_5_IN( ) (bit_clear( DDRB, PB0 ))
#define KEYPAD_5_PU( ) (bit_set( PORTB, PB0 ))
#define KEYPAD_5( ) (bit_is_clear( PINB, PB0 ))
#define KEYPAD_6_IN( ) (bit_clear( DDRE, PE3 ))
#define KEYPAD_6_PU( ) (bit_set( PORTE, PE3 ))
#define KEYPAD_6( ) (bit_is_clear( PINE, PE3 ))
#define KEYPAD_7_IN( ) (bit_clear( DDRB, PB6 ))
#define KEYPAD_7_PU( ) (bit_set( PORTB, PB6 ))
#define KEYPAD_7( ) (bit_is_clear( PINB, PB6 ))
#define KEYPAD_8_IN( ) (bit_clear( DDRB, PB2 ))
#define KEYPAD_8_PU( ) (bit_set( PORTB, PB2 ))
#define KEYPAD_8( ) (bit_is_clear( PINB, PB2 ))
#define KEYPAD_9_IN( ) (bit_clear( DDRE, PE5 ))
#define KEYPAD_9_PU( ) (bit_set( PORTE, PE5 ))
#define KEYPAD_9( ) (bit_is_clear( PINE, PE5 ))
#define KEYPAD_STAR_IN( ) (bit_clear( DDRB, PB7 ))
#define KEYPAD_STAR_PU( ) (bit_set( PORTB, PB7 ))
#define KEYPAD_STAR( ) (bit_is_clear( PINB, PB7 ))
#define KEYPAD_HASH_IN( ) (bit_clear( DDRE, PE6 ))
#define KEYPAD_HASH_PU( ) (bit_set( PORTE, PE6 ))
#define KEYPAD_HASH( ) (bit_is_clear( PINE, PE6 ))

//IO pins of keypad LEDs
#define KEYPAD_GREEN_OUT( ) (bit_set( DDRG, PG0 ))
#define KEYPAD_GREEN_ACT( ) (bit_set( PORTG, PG0 ))
#define KEYPAD_GREEN_IDLE( ) (bit_clear( PORTG, PG0 ))
#define KEYPAD_YELLOW_OUT( ) (bit_set( DDRG, PG1 ))
#define KEYPAD_YELLOW_ACT( ) (bit_set( PORTG, PG1 ))
#define KEYPAD_YELLOW_IDLE( ) (bit_clear( PORTG, PG1 ))

//variables for debouncing keys (bit 0 = 0, bit 1 = 1, ..., bis 9 = 9, bit 10 = *, bit 11 = #)
unsigned short KeypadLastStates[5]; //last states of keys
#define KEYPAD_MASK_0 0x0001
#define KEYPAD_MASK_1 0x0002
#define KEYPAD_MASK_2 0x0004
#define KEYPAD_MASK_3 0x0008
#define KEYPAD_MASK_4 0x0010
#define KEYPAD_MASK_5 0x0020
#define KEYPAD_MASK_6 0x0040
#define KEYPAD_MASK_7 0x0080
#define KEYPAD_MASK_8 0x0100
#define KEYPAD_MASK_9 0x0200
#define KEYPAD_MASK_STAR 0x0400
#define KEYPAD_MASK_HASH 0x0800

//key states
unsigned short KeypadState, KeypadPressed, KeypadReleased;

//key code entered so far (* <code> #)
unsigned char KeypadCode[32]; //entered code in BCD, MSB first
unsigned char KeypadCodeLen; //0xFF if not entering a code

//initialize
void KeypadInit( void ) //(extern)
{
  unsigned char i;

  //initialize I/O pins
  KEYPAD_0_IN( );
  KEYPAD_0_PU( );
  KEYPAD_1_IN( );
  KEYPAD_1_PU( );
  KEYPAD_2_IN( );
  KEYPAD_2_PU( );
  KEYPAD_3_IN( );
  KEYPAD_3_PU( );
  KEYPAD_4_IN( );
  KEYPAD_4_PU( );
  KEYPAD_5_IN( );
  KEYPAD_5_PU( );
  KEYPAD_6_IN( );
  KEYPAD_6_PU( );
  KEYPAD_7_IN( );
  KEYPAD_7_PU( );
  KEYPAD_8_IN( );
  KEYPAD_8_PU( );
  KEYPAD_9_IN( );
  KEYPAD_9_PU( );
  KEYPAD_STAR_IN( );
  KEYPAD_STAR_PU( );
  KEYPAD_HASH_IN( );
  KEYPAD_HASH_PU( );
  KEYPAD_GREEN_OUT( );
  KEYPAD_YELLOW_OUT( );

  //initialize keypad variables
  for( i = 0; i < count( KeypadLastStates ); i++ )
    KeypadLastStates[i] = 0x0000;
  KeypadState = 0x0000;
  KeypadPressed = 0x0000;
  KeypadReleased = 0x0000;

  //status of keypad: idle
  KeypadCodeLen = 0xFF;
  KEYPAD_GREEN_IDLE( );
  KEYPAD_YELLOW_IDLE( );
}

//tick procedure - call every 20ms
void KeypadTick20( void ) //(extern)
{
  unsigned short CurrentState, Pressed, Released;
  unsigned char i;

  //read current key state from port pins
  CurrentState = 0;
  if( KEYPAD_0( ) ) CurrentState |= KEYPAD_MASK_0;
  if( KEYPAD_1( ) ) CurrentState |= KEYPAD_MASK_1;
  if( KEYPAD_2( ) ) CurrentState |= KEYPAD_MASK_2;
  if( KEYPAD_3( ) ) CurrentState |= KEYPAD_MASK_3;
  if( KEYPAD_4( ) ) CurrentState |= KEYPAD_MASK_4;
  if( KEYPAD_5( ) ) CurrentState |= KEYPAD_MASK_5;
  if( KEYPAD_6( ) ) CurrentState |= KEYPAD_MASK_6;
  if( KEYPAD_7( ) ) CurrentState |= KEYPAD_MASK_7;
  if( KEYPAD_8( ) ) CurrentState |= KEYPAD_MASK_8;
  if( KEYPAD_9( ) ) CurrentState |= KEYPAD_MASK_9;
  if( KEYPAD_STAR( ) ) CurrentState |= KEYPAD_MASK_STAR;
  if( KEYPAD_HASH( ) ) CurrentState |= KEYPAD_MASK_HASH;

  //collect entropy
  if( KeypadLastStates[0] != CurrentState ) //change on keypad input pin
    Timing20Entropy( ); //get entropy from current 20ms tick counter

  //get debounced key state
  // - move old states one place back and save current state
  // - keys pressed all the time are pressed
  // - keys released all the time are released
  Pressed = 0xFFFF;
  Released = 0x0000;
  for( i = count( KeypadLastStates ) - 1; i > 0; i-- )
  {
    Pressed &= KeypadLastStates[i];
    Released |= KeypadLastStates[i];
    KeypadLastStates[i] = KeypadLastStates[i-1];
  }
  Pressed &= CurrentState;
  Released |= CurrentState;
  KeypadLastStates[0] = CurrentState;
  Pressed &= ~KeypadState;
  Released = ~Released & KeypadState;
  KeypadPressed |= Pressed;
  KeypadReleased |= Released;
  KeypadState |= Pressed;
  KeypadState &= ~Released;

  //output released and pressed keys in debug mode
  if( Released & KEYPAD_MASK_0 ) debug_keys_printf( "0 up" );
  if( Released & KEYPAD_MASK_1 ) debug_keys_printf( "1 up" );
  if( Released & KEYPAD_MASK_2 ) debug_keys_printf( "2 up" );
  if( Released & KEYPAD_MASK_3 ) debug_keys_printf( "3 up" );
  if( Released & KEYPAD_MASK_4 ) debug_keys_printf( "4 up" );
  if( Released & KEYPAD_MASK_5 ) debug_keys_printf( "5 up" );
  if( Released & KEYPAD_MASK_6 ) debug_keys_printf( "6 up" );
  if( Released & KEYPAD_MASK_7 ) debug_keys_printf( "7 up" );
  if( Released & KEYPAD_MASK_8 ) debug_keys_printf( "8 up" );
  if( Released & KEYPAD_MASK_9 ) debug_keys_printf( "9 up" );
  if( Released & KEYPAD_MASK_STAR ) debug_keys_printf( "* up" );
  if( Released & KEYPAD_MASK_HASH ) debug_keys_printf( "# up" );
  if( Pressed & KEYPAD_MASK_0 ) debug_keys_printf( "0 down" );
  if( Pressed & KEYPAD_MASK_1 ) debug_keys_printf( "1 down" );
  if( Pressed & KEYPAD_MASK_2 ) debug_keys_printf( "2 down" );
  if( Pressed & KEYPAD_MASK_3 ) debug_keys_printf( "3 down" );
  if( Pressed & KEYPAD_MASK_4 ) debug_keys_printf( "4 down" );
  if( Pressed & KEYPAD_MASK_5 ) debug_keys_printf( "5 down" );
  if( Pressed & KEYPAD_MASK_6 ) debug_keys_printf( "6 down" );
  if( Pressed & KEYPAD_MASK_7 ) debug_keys_printf( "7 down" );
  if( Pressed & KEYPAD_MASK_8 ) debug_keys_printf( "8 down" );
  if( Pressed & KEYPAD_MASK_9 ) debug_keys_printf( "9 down" );
  if( Pressed & KEYPAD_MASK_STAR ) debug_keys_printf( "* down" );
  if( Pressed & KEYPAD_MASK_HASH ) debug_keys_printf( "# down" );
}

//task function to do the work - call from main loop
void KeypadTask( void ) //(extern)
{
  unsigned short mask;
  unsigned char i;

  //star was pressed
  if( KeypadPressed & KEYPAD_MASK_STAR )
  {
    KeypadPressed &= ~KEYPAD_MASK_STAR;
    //not entering a code
    if( KeypadCodeLen >> 1 > count( KeypadCode ) )
    {
      //start entering a code
      KeypadCodeLen = 0;
      KEYPAD_YELLOW_ACT( );
    }
    //entering a code
    else
    {
      //abort entering a code
      KeypadCodeLen = 0xFF;
      KEYPAD_YELLOW_IDLE( );
    }
  }

  //not entering a code
  if( KeypadCodeLen >> 1 > count( KeypadCode ) )
  {
    //keys 1..8 :if a key is pressed - toggle output
    for( i = 1, mask = KEYPAD_MASK_1; i <= 8; i++, mask <<= 1 )
    {
      if( KeypadPressed & mask )
      {
        KeypadPressed &= ~mask;
        OutputChange( -1, i ); //toggle output
      }
    }
    //key 9: turn on all outputs
    if( KeypadPressed & KEYPAD_MASK_9 )
    {
      KeypadPressed &= ~KEYPAD_MASK_9;
      OutputChangeAll( 1 );
    }
    //keys 0: turn off all outputs
    if( KeypadPressed & KEYPAD_MASK_0 )
    {
      KeypadPressed &= ~KEYPAD_MASK_0;
      OutputChangeAll( 0 );
    }
  }
  //entering a code
  else
  {
    //add pressed key to code (this sorts keys by value if multiple keys are pressed at the same time)
    for( i = 0, mask = KEYPAD_MASK_0; i <= 9; i++, mask <<= 1 )
    {
      if( KeypadPressed & mask )
      {
        KeypadPressed &= ~mask;
        //buffer not yet full
        if( KeypadCodeLen >> 1 < count( KeypadCode ) )
        {
          if( (KeypadCodeLen & 0x01) == 0x00 ) //put key into high nibble
            KeypadCode[KeypadCodeLen >> 1] = i << 4;
          else //put key into low nibble
            KeypadCode[KeypadCodeLen >> 1] |= i;
          KeypadCodeLen++;
        }
      }
    }
  }

  //hash was pressed
  if( KeypadPressed & KEYPAD_MASK_HASH )
  {
    KeypadPressed &= ~KEYPAD_MASK_HASH;
    //entering a code
    if( KeypadCodeLen >> 1 <= count( KeypadCode ) )
    {
      //process entered code
      if( CodeProc( KeypadCode, KeypadCodeLen ) == 0x00 )
        //set acknowledge if successful
        KEYPAD_GREEN_ACT( );
      //end entering a code
      KeypadCodeLen = 0xFF;
      KEYPAD_YELLOW_IDLE( );
    }
  }

  //hash was released
  if( KeypadReleased & KEYPAD_MASK_HASH )
  {
    KeypadReleased &= ~KEYPAD_MASK_HASH;
    //take back acknowledge
    KEYPAD_GREEN_IDLE( );
  }
}
