/* 8BitAmEthernet
 * version 0.3.2 date 2006-04-08
 * Copyright 2005-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 * a project of CCC-AC - http://www.cccac.de/
 */

#include "code.h"
#include "config.h"
#include "debug.h"
#include "eeprom.h"
#include "macros.h"
#include "output.h"
#include "rtl8019.h"

//convert 3 BCD digits to a byte
static inline unsigned char Code3Bcd2Byte( unsigned char * pCode, unsigned char Pos )
{
  //byte starts at even position
  if( (Pos & 0x01) == 0x00 )
    return (pCode[Pos>>1] >> 4) * 100 + (pCode[Pos>>1] & 0x0F) * 10 + (pCode[(Pos>>1)+1] >> 4);
  //byte starts at odd position
  else
    return (pCode[Pos>>1] & 0x0F) * 100 + (pCode[(Pos>>1)+1] >> 4) * 10 + (pCode[(Pos>>1)+1] & 0x0F);
}

//convert 5 BCD digits to a word
static inline unsigned short Code5Bcd2Word( unsigned char * pCode, unsigned char Pos )
{
  //word starts at even position
  if( (Pos & 0x01) == 0x00 )
    return (unsigned short)(pCode[Pos>>1] >> 4) * 10000
         + (unsigned short)(pCode[Pos>>1] & 0x0F) * 1000
         + (unsigned short)(pCode[(Pos>>1)+1] >> 4) * 100
         + (unsigned short)(pCode[(Pos>>1)+1] & 0x0F) * 10
         + (unsigned short)(pCode[(Pos>>1)+2] >> 4);
  //word starts at odd position
  else
    return (unsigned short)(pCode[Pos>>1] & 0x0F) * 10000
         + (unsigned short)(pCode[(Pos>>1)+1] >> 4) * 1000
         + (unsigned short)(pCode[(Pos>>1)+1] & 0x0F) * 100
         + (unsigned short)(pCode[(Pos>>1)+2] >> 4) * 10
         + (unsigned short)(pCode[(Pos>>1)+2] & 0x0F);
}

//set MAC address
//code is stored in BCD, MSB first - magic is already removed
//returns 0x00 on success
static inline unsigned char CodeSetMac( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned char b4, b5;

  //invalid length
  if( CodeLen != 8 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  debug_code_printf( "mac" );

  //get last 2 bytes from code
  b4 = Code3Bcd2Byte( pCode, 2 );
  b5 = Code3Bcd2Byte( pCode, 5 );

  //set MAC address to 02:CC:C0:AC:__:__
  ConfigMac[0] = 0x02;
  ConfigMac[1] = 0xCC;
  ConfigMac[2] = 0xC0;
  ConfigMac[3] = 0xAC;
  ConfigMac[4] = b4;
  ConfigMac[5] = b5;
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteMac;

  //re-initialize RTL8019
  RtlReinit( );

  return 0x00;
}

//set IP address
//code is stored in BCD, MSB first - magic is already removed
//returns 0x00 on success
static inline unsigned char CodeSetIp( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned char b0, b1, b2, b3;

  //invalid length
  if( CodeLen != 14 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  debug_code_printf( "ip" );

  //get 4 bytes from code
  b0 = Code3Bcd2Byte( pCode, 2 );
  b1 = Code3Bcd2Byte( pCode, 5 );
  b2 = Code3Bcd2Byte( pCode, 8 );
  b3 = Code3Bcd2Byte( pCode, 11 );

  //set IP address
  ConfigIp[0] = b0;
  ConfigIp[1] = b1;
  ConfigIp[2] = b2;
  ConfigIp[3] = b3;
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteIp;

  return 0x00;
}

//set subnet mask
//code is stored in BCD, MSB first - magic is already removed
//returns 0x00 on success
static inline unsigned char CodeSetMask( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned char b[4];
  unsigned long l;

  //invalid length
  if( CodeLen != 14 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  debug_code_printf( "subnet mask" );

  //get 4 bytes from code
  b[0] = Code3Bcd2Byte( pCode, 2 );
  b[1] = Code3Bcd2Byte( pCode, 5 );
  b[2] = Code3Bcd2Byte( pCode, 8 );
  b[3] = Code3Bcd2Byte( pCode, 11 );

  //check subnet mask (subnet mask must be 1..10..0 in binary)
  l = (unsigned long)b[0] << 24 | (unsigned long)b[1] << 16 | (unsigned long)b[2] << 8 | (unsigned long)b[3];
  while( l & 0x80000000 ) //shift out 1 bits
    l <<= 1;
  if( l != 0x00000000 ) //if not every bit is 0 ...
    return 0xFF; //... there must have been a 1 bit after a 0 bit (invalid netmask)

  //set subnet mask
  ConfigMask[0] = b[0];
  ConfigMask[1] = b[1];
  ConfigMask[2] = b[2];
  ConfigMask[3] = b[3];
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteMask;

  return 0x00;
}

//set gateway IP address
//code is stored in BCD, MSB first - magic is already removed
//returns 0x00 on success
static inline unsigned char CodeSetGw( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned char b0, b1, b2, b3;

  //invalid length
  if( CodeLen != 14 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  debug_code_printf( "gateway ip" );

  //get 4 bytes from code
  b0 = Code3Bcd2Byte( pCode, 2 );
  b1 = Code3Bcd2Byte( pCode, 5 );
  b2 = Code3Bcd2Byte( pCode, 8 );
  b3 = Code3Bcd2Byte( pCode, 11 );

  //set gateway IP address
  ConfigGw[0] = b0;
  ConfigGw[1] = b1;
  ConfigGw[2] = b2;
  ConfigGw[3] = b3;
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteGw;

  return 0x00;
}

//set port for S8P
//code is stored in BCD, MSB first - magic is already removed
//returns 0x00 on success
static inline unsigned char CodeSetS8pPort( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned short port;

  //invalid length
  if( CodeLen != 7 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }
  
  debug_code_printf( "s8p port" );

  //get port number from from code
  port = Code5Bcd2Word( pCode, 2 );

  //save port for S8P
  ConfigS8pPort = port;
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteS8pPort;

  return 0x00;
}

//set symmetric key for S8P
//code is stored in BCD, MSB first - magic is already removed
//returns 0x00 on success
static inline unsigned char CodeSetS8pKey( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned char key[16], i, j;

  //invalid length
  if( CodeLen != 2 + count( key ) * 3 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  debug_code_printf( "s8p key" );

  //get bytes from code
  for( i = 0, j = 2; i < count( key ); i++, j += 3 )
    key[i] = Code3Bcd2Byte( pCode, j );

  //set symmetric key
  for( i = 0; i < min( count( key ), count( ConfigS8pKey ) ); i++ )
    ConfigS8pKey[i] = key[i];
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteS8pKey;

  return 0x00;
}

//set port for webserver
//code is stored in BCD, MSB first - magic is already removed
//returns 0x00 on success
static inline unsigned char CodeSetHttpPort( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned short port;

  //invalid length
  if( CodeLen != 7 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }
  
  debug_code_printf( "http port" );

  //get port number from from code
  port = Code5Bcd2Word( pCode, 2 );

  //save port for webserver
  ConfigHttpPort = port;
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteHttpPort;

  return 0x00;
}

//save default output state
static inline unsigned char CodeSaveDefOut( unsigned char * pCode, unsigned char CodeLen )
{
  //invalid length
  if( CodeLen != 2 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  debug_code_printf( "save default output" );

  //save default output state (atomically because only a single assignment)
  ConfigDefOut = OutputGetState( );
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteDefOut;

  return 0x00;
}

//set default output state
static inline unsigned char CodeSetDefOut( unsigned char * pCode, unsigned char CodeLen )
{
  unsigned char i, digit, defOut;

  //invalid length
  if( CodeLen != 10 ) 
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  debug_code_printf( "set default output" );

  //parse default output state into a single byte
  //8 times "0" or "1"
  defOut = 0;
  for( i = 1; i < 5; i++ )
  {
    defOut >>= 1; //next digit
    digit = pCode[i] >> 4; //1st digit in byte
    if( digit != 0 && digit != 1 )
      return 0xFF;
    if( digit == 1 )
      defOut |= 0x80;
    defOut >>= 1; //next digit
    digit = pCode[i] & 0x0F; //2nd digit in byte
    if( digit != 0 && digit != 1 )
      return 0xFF;
    if( digit == 1 )
      defOut |= 0x80;
  }

  //store default output state (atomically because only a single assignment)
  ConfigDefOut = defOut;
  //schedule write to EEPROM
  EepromNeedWrite |= EepromNeedWriteDefOut;

  return 0x00;
}

//process a code
//code is stored in BCD, MSB first
//returns 0x00 on success
unsigned char CodeProc( unsigned char * pCode, unsigned char CodeLen )
{
  //check green LED
  if( CodeLen == 0 )
  {
    debug_code_printf( "empty" );
    return 0x00;
  }

  //not long enough for magic
  if( CodeLen < 2 )
  {
    debug_code_printf( "unknown" );
    return 0xFF;
  }

  //branch according to magic
  switch( pCode[0] )
  {
    case 0x00: return CodeSetMac( pCode, CodeLen ); //set MAC address
    case 0x10: return CodeSetIp( pCode, CodeLen ); //set IP address
    case 0x11: return CodeSetMask( pCode, CodeLen ); //set subnet mask address
    case 0x12: return CodeSetGw( pCode, CodeLen ); //set gateway IP address
    case 0x20: return CodeSetS8pPort( pCode, CodeLen ); //set port for S8P
    case 0x21: return CodeSetS8pKey( pCode, CodeLen ); //set symmetric key for S8P
    case 0x30: return CodeSetHttpPort( pCode, CodeLen ); //set port for webserver
    case 0x40: return CodeSaveDefOut( pCode, CodeLen ); //save default output state
    case 0x41: return CodeSetDefOut( pCode, CodeLen ); //set default output state
  }

  debug_code_printf( "unknown" );

  //unknown code
  return 0xFF;
}

