/* 8BitAmEthernet
 * version 0.2 date 2005-03-07
 * Copyright 2005 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 * a project of CCC-AC - http://www.cccac.de/
 */

#include <string.h>
#include <avr/pgmspace.h>

#include "http.h"
#include "macros.h"
#include "output.h"
#include "tcp.h"

//receive states while parsing HTTP request
#define HTTP_RCV_CMD 0 //receiving HTTP command (e.g. GET)
#define HTTP_RCV_FILE 1 //receiving filename
#define HTTP_RCV_HDR_END 2 //receiving header until end
#define HTTP_RCV_DONE 3 //receiving completed

//knwon HTTP commands
#define HTTP_CMD_UNKNOWN 0 //unknown HTTP command
#define HTTP_CMD_GET 1 //GET command

//files available using HTTP
#define HTTP_FILE_NOT_FOUND 0 //file not found file
#define HTTP_FILE_INDEX 1 //index file
#define HTTP_FILE_ON 2 //turn on links (+0..+7)
#define HTTP_FILE_OFF 10 //turn off links (+0..+7)
#define HTTP_FILE_IMG1 18 //image: on
#define HTTP_FILE_IMG0 19 //image: off

//table with HTTP connections
struct HttpConnection
{
  unsigned char ConnNo; //number of the TCP connection, 0xFF is unused
  unsigned long RcvCnt; //number of received bytes on connection
  unsigned char RcvState; //what is being received at the moment
  unsigned char RcvBuf[16]; //receive buffer
  unsigned char RcvBufLen; //length of data in receive buffer
  unsigned char Command; //the command the client issued
  char PreHttp1; //flag if a HTTP version before 1.0 is used
  unsigned char File; //the file the client requested
  uint16_t pSndData; //the data to send (progmem pointer)
  unsigned short SndDataLen; //the length of the data to send
  char Vars; //flag if to interpret variables
} HttpConns[4];

//data to send
// - characters from 0x80..0xFF indicate 1 character of a variable
//   - e.g. "\x80\x80\x80" indicates to insert the first 3 characters of varibale 0x80
//   - for meaning of variables, see HttpGetVariable
#define HttpBadRequestHeaderSize 121
const char PROGMEM HttpBadRequest[] =
"HTTP/1.0 400 Bad Request\r\n"
"Server: 8BitAmEthernet (ATMEGA128) by CCC-AC.de\r\n"
"Connection: close\r\n"
"Content-Type: text/html\r\n"
"\r\n"
"<html>\r\n"
"  <head><title>8BitAmEthernet - 400 Bad Request</title></head>\r\n"
"  <body>\r\n"
"    <h1>8BitAmEthernet - 400 Bad Request</h1>\r\n"
"    Your browser sent a request that this server could not understand.<br>\r\n"
"  </body>\r\n"
"</html>\r\n";
#define HttpNotFoundHeaderSize 119
const char PROGMEM HttpNotFound[] =
"HTTP/1.0 404 Not Found\r\n"
"Server: 8BitAmEthernet (ATMEGA128) by CCC-AC.de\r\n"
"Connection: close\r\n"
"Content-Type: text/html\r\n"
"\r\n"
"<html>\r\n"
"  <head><title>8BitAmEthernet - 404 Not Found</title></head>\r\n"
"  <body>\r\n"
"    <h1>8BitAmEthernet - 404 Not Found</h1>\r\n"
"    The requested URL was not found on this server.<br>\r\n"
"  </body>\r\n"
"</html>\r\n";
#define HttpIndexHeaderSize 112
const char PROGMEM HttpIndex[] =
"HTTP/1.0 200 Ok\r\n"
"Server: 8BitAmEthernet (ATMEGA128) by CCC-AC.de\r\n"
"Connection: close\r\n"
"Content-Type: text/html\r\n"
"\r\n"
"<html>\r\n"
"  <head>\r\n"
"    <title>8BitAmEthernet</title>\r\n"
"    <meta http-equiv=\"pragma\" content=\"no-cache\">\r\n"
"  </head>\r\n"
"  <body bgcolor=\"#DCDCDC\" text=\"#000000\" link=\"#0000C0\" alink=\"#0000C0\" vlink=\"#0000C0\">\r\n"
"    <h1>8BitAmEthernet</h1>\r\n"
"      <table border=\"1\" cellpadding=\"3\">\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>port</b></td>\r\n"
"          <td align=\"center\"><b>state</b></td>\r\n"
"          <td align=\"center\"><b>actions</b></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>1</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x91\" alt=\"\x81\x81\x81\">&nbsp;\x81\x81\x81</td>\r\n"
"          <td align=\"center\"><a href=\"/on1\">on</a>&nbsp;&nbsp;<a href=\"/off1\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>2</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x92\" alt=\"\x82\x82\x82\">&nbsp;\x82\x82\x82</td>\r\n"
"          <td align=\"center\"><a href=\"/on2\">on</a>&nbsp;&nbsp;<a href=\"/off2\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>3</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x93\" alt=\"\x83\x83\x83\">&nbsp;\x83\x83\x83</td>\r\n"
"          <td align=\"center\"><a href=\"/on3\">on</a>&nbsp;&nbsp;<a href=\"/off3\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>4</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x94\" alt=\"\x84\x84\x84\">&nbsp;\x84\x84\x84</td>\r\n"
"          <td align=\"center\"><a href=\"/on4\">on</a>&nbsp;&nbsp;<a href=\"/off4\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>5</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x95\" alt=\"\x85\x85\x85\">&nbsp;\x85\x85\x85</td>\r\n"
"          <td align=\"center\"><a href=\"/on5\">on</a>&nbsp;&nbsp;<a href=\"/off5\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>6</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x96\" alt=\"\x86\x86\x86\">&nbsp;\x86\x86\x86</td>\r\n"
"          <td align=\"center\"><a href=\"/on6\">on</a>&nbsp;&nbsp;<a href=\"/off6\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>7</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x97\" alt=\"\x87\x87\x87\">&nbsp;\x87\x87\x87</td>\r\n"
"          <td align=\"center\"><a href=\"/on7\">on</a>&nbsp;&nbsp;<a href=\"/off7\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"        <tr>\r\n"
"          <td align=\"center\"><b>8</b></td>\r\n"
"          <td align=\"center\"><img src=\"/img\x98\" alt=\"\x88\x88\x88\">&nbsp;\x88\x88\x88</td>\r\n"
"          <td align=\"center\"><a href=\"/on8\">on</a>&nbsp;&nbsp;<a href=\"/off8\">&nbsp;off</a></td>\r\n"
"        </tr>\r\n"
"      </table>\r\n"
"      <br><br>\r\n"
"      <a href=\"/\">refresh</a>\r\n"
"  </body>\r\n"
"</html>\r\n";
#define HttpImg1HeaderSize 112
const char PROGMEM HttpImg1[] =
"HTTP/1.0 200 Ok\r\n"
"Server: 8BitAmEthernet (ATMEGA128) by CCC-AC.de\r\n"
"Connection: close\r\n"
"Content-Type: image/png\r\n"
"\r\n"
"\x89\x50\x4E\x47\x0D\x0A\x1A\x0A\x00\x00\x00\x0D\x49\x48\x44\x52"
"\x00\x00\x00\x10\x00\x00\x00\x10\x08\x02\x00\x00\x00\x90\x91\x68"
"\x36\x00\x00\x02\x12\x49\x44\x41\x54\x28\xCF\x95\x92\xCF\x6A\x1A"
"\x51\x14\xC6\xBF\xDC\x7B\xC7\x89\x73\x47\x1D\x15\x62\x89\x50\x12"
"\x34\x9A\x59\x34\x7F\x30\x8B\x24\xCB\xD0\x6D\x1F\x20\xE4\x71\xDA"
"\x77\xF1\x01\x0A\x59\x84\x86\xEE\x03\x81\x80\x8A\x09\xB9\x0C\x1D"
"\x5A\x3A\x3A\x63\xC4\x1B\xC7\x99\x51\xC7\xA1\x0B\x25\x6D\xBA\x6A"
"\xCF\xEE\xC0\xF9\xF8\xCE\xF9\xCE\x6F\x4D\x08\x81\xFF\x29\xF6\x67"
"\xF3\xFC\xFC\x7C\x7F\x7F\x2F\x84\x18\x0E\x87\x8C\xB1\x8D\x8D\x8D"
"\xED\xED\xED\xAD\xAD\xAD\x7C\x3E\xFF\x32\xB3\xF6\xE2\xD0\x6E\xB7"
"\xAF\xAF\xAF\x7D\xDF\x5F\x5F\x5F\x57\x55\x95\x31\xA6\xEB\x7A\x2A"
"\x95\x52\x55\xB5\x56\xAB\x99\xA6\xF9\xCA\xA1\xD5\x6A\x35\x9B\x4D"
"\x4A\x69\x2A\x95\xEA\x1D\xF6\xFA\x47\x7D\xCE\x78\xF1\x67\xD1\xFC"
"\x66\xEA\x33\xBD\xDF\xEF\x03\x58\x6A\x18\x00\x29\xE5\xE5\xE5\x25"
"\x80\xC0\x08\xAC\x4F\xD6\x66\x61\x73\x17\xBB\x11\xA2\xF0\x5D\x78"
"\x85\xAB\xD3\x2F\xA7\xE6\x77\x33\x0C\xC3\xD1\x68\x64\x18\x06\x03"
"\xD0\xE9\x74\xC6\xE3\x31\x00\xE7\xA3\x73\x5C\x38\xAE\xA3\x9E\x43"
"\xCE\x87\xEF\xC0\xB1\x60\xDD\xBC\xBF\x29\x7F\x2D\x1B\x23\x43\x4A"
"\xB9\x12\x3C\x3C\x3C\xA8\xAA\x3A\x38\x1A\x94\x8B\xE5\x3A\xEA\xFB"
"\xD8\x27\x20\x01\x02\x05\xCA\x1C\xF3\x08\x51\xDB\x6C\xEF\x75\xF7"
"\x96\xCB\x13\x00\x9E\xE7\xA9\xAA\x3A\x38\x1E\xE4\x91\xCF\x21\x47"
"\x40\x08\x08\x05\x65\x60\x14\x54\x83\xD6\x7D\xD3\xE5\x9C\x67\x32"
"\x99\xD5\x0D\x94\x52\x00\x9C\xF2\x08\x91\x0F\x3F\x40\x40\x41\x23"
"\x44\x09\x92\x97\x34\x39\xE7\x49\x92\xAC\x04\xA5\x52\xC9\xF7\x7D"
"\xE3\x87\x11\x1E\x86\x0E\x1C\x06\xA6\x40\x49\x90\x0C\x30\x88\x10"
"\x4D\x31\x35\xE7\x66\xB1\x58\x04\x30\x99\x4C\x08\x80\x6A\xB5\xAA"
"\x28\x4A\xE5\xB1\xE2\xC2\xB5\x60\x59\xB0\x04\x84\x05\xCB\x85\xEB"
"\xC2\x95\x90\x67\xCA\x59\x26\x93\x61\x8C\xAD\x1C\xAA\xD5\xAA\x10"
"\x82\x87\xFC\xE0\xF3\x41\xE7\x43\x67\x8A\x69\x1A\x69\x00\x53\x4C"
"\x25\x64\x03\x8D\x73\x9C\x83\x63\x3E\x9F\xFF\xFE\xF4\xDD\xDD\x9D"
"\x6D\xDB\x84\x10\xBF\xE2\x8B\x7D\x21\xDE\x0A\x00\x3B\xC1\xCE\xC9"
"\xE2\xE4\x82\x5C\x70\xCE\xE3\x38\xB6\x6D\xFB\x15\x1A\xB7\xB7\xB7"
"\x52\x4A\xC6\x58\x36\x9B\xD5\x34\x4D\xD3\xB4\x42\xA1\xA0\xEB\x7A"
"\x3A\x9D\xD6\x75\xFD\xE9\xE9\x29\x8E\xE3\x57\xF0\x35\x1A\x0D\xCF"
"\xF3\x7A\xBD\xDE\x62\xB1\x20\x84\x2C\x33\x99\xCD\x66\x8C\xB1\x25"
"\x1A\x7F\xC3\xF7\x8F\xF5\x0B\x9C\x04\xE8\x39\xD4\x23\x69\x49\x00"
"\x00\x00\x00\x49\x45\x4E\x44\xAE\x42\x60\x82";
#define HttpImg0HeaderSize 112
const char PROGMEM HttpImg0[] =
"HTTP/1.0 200 Ok\r\n"
"Server: 8BitAmEthernet (ATMEGA128) by CCC-AC.de\r\n"
"Connection: close\r\n"
"Content-Type: image/png\r\n"
"\r\n"
"\x89\x50\x4E\x47\x0D\x0A\x1A\x0A\x00\x00\x00\x0D\x49\x48\x44\x52"
"\x00\x00\x00\x10\x00\x00\x00\x10\x08\x02\x00\x00\x00\x90\x91\x68"
"\x36\x00\x00\x01\xF4\x49\x44\x41\x54\x28\xCF\x95\x92\xCB\x4A\x23"
"\x51\x10\x86\xBF\x3E\x7D\x92\xD3\xC9\xE9\x34\xAD\xB9\xC8\xCC\x38"
"\xE3\x85\x10\x10\x04\x85\x20\xB8\x1F\xE6\x01\x5C\xB8\x12\xB7\xEE"
"\x05\x9F\xC6\x17\xF0\x0D\x14\x17\xC6\x85\x2B\x17\x82\x10\x85\x08"
"\x0D\x12\x94\xA1\x35\xD7\x4E\xD2\x86\x89\x9D\x38\x8B\x44\x1D\x67"
"\x35\x53\x14\x05\x05\xF5\x51\x3F\x55\xBF\xE1\x79\x1E\xFF\x13\xF2"
"\xCF\xA6\xD3\xE9\x54\x2A\x15\xCF\xF3\x9A\xCD\xA6\x94\x32\x97\xCB"
"\x2D\x2C\x2C\xCC\xCF\xCF\x4F\x4D\x4D\xBD\xCD\x18\x6F\x1B\xAE\xAE"
"\xAE\x4E\x4E\x4E\x7A\xBD\x9E\x65\x59\x4A\x29\x29\xA5\x6D\xDB\xF1"
"\x78\x5C\x29\x55\x28\x14\x96\x96\x96\x3E\x6C\x28\x97\xCB\x07\x07"
"\x07\xA6\x69\xC6\xE3\x71\x2F\xF2\x6E\x9F\x6F\xA5\x92\x73\x2F\x73"
"\x6B\x99\x35\x3B\x66\x3F\x3C\x3C\x00\x63\x46\x02\x41\x10\x1C\x1E"
"\x1E\x02\xDD\x51\xF7\x54\x9D\x86\xA9\x10\x0B\x22\x2A\x51\xE5\xF8"
"\xEE\x78\xE7\xDB\xCE\xBA\xBD\xDE\xEF\xF7\xDB\xED\xB6\xEB\xBA\x02"
"\xB8\xBE\xBE\xEE\x76\xBB\x40\xC9\x29\x85\x9F\x43\x16\x61\x11\xE6"
"\x20\x07\x0E\xFB\xF7\xFB\xFE\xC8\x97\x52\x06\x41\x00\x08\xE0\xE6"
"\xE6\x46\x29\xE5\x27\xFC\x27\xE7\x89\x2C\x7C\x81\x2C\xE4\xC0\x05"
"\x07\x12\x1C\xF9\x47\xAE\xEB\x1A\x86\x31\x91\x54\xAB\xD5\x94\x52"
"\x77\xCF\x77\x58\x60\x81\x00\x01\xC6\x6B\x95\x9C\xFD\x3C\xD3\x5A"
"\x0B\x21\x26\x80\x69\x9A\x80\x69\x98\x44\xF0\x0B\x06\x60\x40\x04"
"\x2F\xEF\x17\xD7\x5A\x8F\x46\xA3\x89\xA4\x99\x99\x19\xAD\xF5\x6C"
"\x6C\x96\x08\xBA\x50\x87\x26\x04\x10\xC2\x10\x86\x14\xBF\x16\xD3"
"\xE9\x74\x36\x9B\x9D\x00\xF9\x7C\x3E\x16\x8B\xAD\xBA\xAB\xF4\xA1"
"\x05\x4D\xA8\x43\x03\x42\x08\x61\xC0\xD6\xDA\x56\x2A\x95\x92\x52"
"\xBE\x03\x4A\xA9\x8C\x95\xD9\xCE\x6C\xD3\x83\x16\x74\x5E\xB3\xCF"
"\xE6\xF2\xE6\xEE\xF7\x5D\xAD\xF5\x18\x98\x7C\xFA\xF2\xF2\xB2\x5A"
"\xAD\x0A\x21\x1E\x79\x2C\xB5\x4A\xE7\xF5\x73\x60\xE5\xD3\xCA\xC6"
"\xF2\xC6\xDE\x8F\x3D\xAD\x75\x14\x45\xD5\x6A\xF5\x83\x35\x2E\x2E"
"\x2E\x82\x20\x90\x52\x3A\x8E\x93\x4C\x26\x93\xC9\xE4\xF4\xF4\xB4"
"\x6D\xDB\x89\x44\xC2\xB6\xED\x46\xA3\x11\x45\xD1\x07\xF3\x15\x8B"
"\xC5\x5A\xAD\xE6\xFB\xFE\x70\x38\x14\x42\x8C\x6F\x32\x18\x0C\xA4"
"\x94\x63\x6B\xFC\x6D\xBE\x7F\x8C\xDF\x43\xE4\xC2\xC6\x45\x5F\x10"
"\xBA\x00\x00\x00\x00\x49\x45\x4E\x44\xAE\x42\x60\x82";

//parse HTTP command
static void HttpParseCommand( struct HttpConnection * pConn )
{
  //GET command
  if( pConn->RcvBufLen == 3 && strncasecmp( pConn->RcvBuf, "GET", 3 ) == 0 )
    pConn->Command = HTTP_CMD_GET;
  //unknown command is the default
  else
    pConn->Command = HTTP_CMD_UNKNOWN;
}

//parse HTTP file
static void HttpParseFile( struct HttpConnection * pConn )
{
  //index file
  if( pConn->RcvBufLen == 1 && strncasecmp( pConn->RcvBuf, "/", 1 ) == 0 )
    pConn->File = HTTP_FILE_INDEX;
  //turn on link
  else if( pConn->RcvBufLen == 4 && strncasecmp( pConn->RcvBuf, "/on", 3 ) == 0 &&
           pConn->RcvBuf[3] >= '1' && pConn->RcvBuf[3] <= '8' )
    pConn->File = HTTP_FILE_ON + pConn->RcvBuf[3] - '1';
  //turn off link
  else if( pConn->RcvBufLen == 5 && strncasecmp( pConn->RcvBuf, "/off", 4 ) == 0 &&
           pConn->RcvBuf[4] >= '1' && pConn->RcvBuf[4] <= '8' )
    pConn->File = HTTP_FILE_OFF + pConn->RcvBuf[4] - '1';
  //image: on
  else if( pConn->RcvBufLen == 5 && strncasecmp( pConn->RcvBuf, "/img1", 5 ) == 0 )
    pConn->File = HTTP_FILE_IMG1;
  //image: off
  else if( pConn->RcvBufLen == 5 && strncasecmp( pConn->RcvBuf, "/img0", 5 ) == 0 )
    pConn->File = HTTP_FILE_IMG0;
  //error file is the default
  else
    pConn->File = HTTP_FILE_NOT_FOUND;
}

//process HTTP request
static void HttpProcessRequest( struct HttpConnection * pConn )
{
  //different commands
  switch( pConn->Command )
  {

    case HTTP_CMD_GET:
      //different actions
      switch( pConn->File )
      {

        case HTTP_FILE_ON+0:
        case HTTP_FILE_ON+1:
        case HTTP_FILE_ON+2:
        case HTTP_FILE_ON+3:
        case HTTP_FILE_ON+4:
        case HTTP_FILE_ON+5:
        case HTTP_FILE_ON+6:
        case HTTP_FILE_ON+7:
          OutputChange( 1, pConn->File - HTTP_FILE_ON + 1 ); //turn on selected output
          pConn->File = HTTP_FILE_INDEX; //"symlink" to index file
          break;

        case HTTP_FILE_OFF+0:
        case HTTP_FILE_OFF+1:
        case HTTP_FILE_OFF+2:
        case HTTP_FILE_OFF+3:
        case HTTP_FILE_OFF+4:
        case HTTP_FILE_OFF+5:
        case HTTP_FILE_OFF+6:
        case HTTP_FILE_OFF+7:
          OutputChange( 0, pConn->File - HTTP_FILE_OFF + 1 ); //turn off selected output
          pConn->File = HTTP_FILE_INDEX; //"symlink" to index file
          break;

      } //switch( pConn->File )

      //different files
      switch( pConn->File )
      {

        case HTTP_FILE_INDEX:
          if( pConn->PreHttp1 )
          {
            pConn->pSndData = (uint16_t)HttpIndex + HttpIndexHeaderSize;
            pConn->SndDataLen = sizeof( HttpIndex ) - 1 - HttpIndexHeaderSize;
          }
          else
          {
            pConn->pSndData = (uint16_t)HttpIndex;
            pConn->SndDataLen = sizeof( HttpIndex ) - 1;
          }
          pConn->Vars = 1;
          break;

        case HTTP_FILE_IMG1:
          if( pConn->PreHttp1 )
          {
            pConn->pSndData = (uint16_t)HttpImg1 + HttpImg1HeaderSize;
            pConn->SndDataLen = sizeof( HttpImg1 ) - 1 - HttpImg1HeaderSize;
          }
          else
          {
            pConn->pSndData = (uint16_t)HttpImg1;
            pConn->SndDataLen = sizeof( HttpImg1 ) - 1;
          }
          pConn->Vars = 0;
          break;

        case HTTP_FILE_IMG0:
          if( pConn->PreHttp1 )
          {
            pConn->pSndData = (uint16_t)HttpImg0 + HttpImg0HeaderSize;
            pConn->SndDataLen = sizeof( HttpImg0 ) - 1 - HttpImg0HeaderSize;
          }
          else
          {
            pConn->pSndData = (uint16_t)HttpImg0;
            pConn->SndDataLen = sizeof( HttpImg0 ) - 1;
          }
          pConn->Vars = 0;
          break;

        case HTTP_FILE_NOT_FOUND:
        default:
          if( pConn->PreHttp1 )
          {
            pConn->pSndData = (uint16_t)HttpNotFound + HttpNotFoundHeaderSize;
            pConn->SndDataLen = sizeof( HttpNotFound ) - 1 - HttpNotFoundHeaderSize;
          }
          else
          {
            pConn->pSndData = (uint16_t)HttpNotFound;
            pConn->SndDataLen = sizeof( HttpNotFound ) - 1;
          }
          pConn->Vars = 1;

      } //switch( pConn->File )
      break;

    case HTTP_CMD_UNKNOWN:
    default:
      if( pConn->PreHttp1 )
      {
        pConn->pSndData = (uint16_t)HttpBadRequest + HttpBadRequestHeaderSize;
        pConn->SndDataLen = sizeof( HttpBadRequest ) - 1 - HttpBadRequestHeaderSize;
      }
      else
      {
        pConn->pSndData = (uint16_t)HttpBadRequest;
        pConn->SndDataLen = sizeof( HttpBadRequest ) - 1;
      }
      pConn->Vars = 1;

  } //switch( pConn->Command )
}

//get a variable
//pBuffer must provide space for at least 16 characters
//returns the length of the variable
static unsigned char HttpGetVariable( unsigned char VarNo, char * pBuffer )
{
  switch( VarNo )
  {

    //current state of output 1..8 ("ON" or "off")
    case 0x81:
    case 0x82:
    case 0x83:
    case 0x84:
    case 0x85:
    case 0x86:
    case 0x87:
    case 0x88:
      if( OutputGet( VarNo - 0x80 ) )
      {
        memcpy( pBuffer, "ON", 2 );
        return 2;
      }
      else
      {
        memcpy( pBuffer, "off", 3 );
        return 3;
      }

    //current state of output 1..8 ("1" or "0")
    case 0x91:
    case 0x92:
    case 0x93:
    case 0x94:
    case 0x95:
    case 0x96:
    case 0x97:
    case 0x98:
      if( OutputGet( VarNo - 0x90 ) )
        pBuffer[0] = '1';
      else
        pBuffer[0] = '0';
        return 1;

    //unknown variable
    default:
      return 0;

  } //switch( VarNo );
}

//called when connection is established
void HttpConnect( unsigned char ConnNo )
{
  unsigned char i;
  struct HttpConnection * pConn;

  //find connection in table (in case TCP calls us twice, this should never happen)
  for( i = 0; i < count( HttpConns ); i++ )
    if( HttpConns[i].ConnNo == ConnNo )
      break;
  //connection not found
  if( i >= count( HttpConns ) )
  {
    //find a free entry
    for( i = 0; i < count( HttpConns ); i++ )
      if( HttpConns[i].ConnNo == 0xFF )
        break;
    if( i >= count( HttpConns ) ) //no free entry found
      return; //ignore this connection (will be closed in first call of HttpSend)
  }
  //get pointer to connection
  pConn = &HttpConns[i];

  //put new connection into table
  pConn->ConnNo = ConnNo;
  pConn->RcvCnt = 0;
  pConn->RcvState = HTTP_RCV_CMD;
  pConn->RcvBufLen = 0;
}

//called when connection is closed / reset
//(after this, the connection number may not be used any more)
void HttpClose( unsigned char ConnNo )
{
  unsigned char i;
  struct HttpConnection * pConn;

  //find connection in table
  for( i = 0; i < count( HttpConns ); i++ )
    if( HttpConns[i].ConnNo == ConnNo )
      break;
  if( i >= count( HttpConns ) ) //connection not found
    return; //ignore this (now already closed) connection
  //get pointer to connection
  pConn = &HttpConns[i];

  //drop connection from table
  pConn->ConnNo = 0xFF;
}

//called when sending data is possible
//(return length of available data, 0xFFFF to close connection)
unsigned short HttpSend( unsigned char ConnNo, unsigned long Pos, unsigned char * pBuffer, unsigned short MaxLen )
{
  unsigned char i, var, VarLen, VarPos;
  struct HttpConnection * pConn;
  unsigned short len, j;
  uint16_t src; //progmem pointer
  char chr, * dest, VarBuffer[16];

  //find connection in table
  for( i = 0; i < count( HttpConns ); i++ )
    if( HttpConns[i].ConnNo == ConnNo )
      break;
  if( i >= count( HttpConns ) ) //connection not found
    return 0xFFFF; //close connection
  //get pointer to connection
  pConn = &HttpConns[i];

  //not done with receiving
  if( pConn->RcvState != HTTP_RCV_DONE )
    return 0; //do not send anything yet

  //at or behind end of data
  if( Pos >= (unsigned long)pConn->SndDataLen )
    return 0xFFFF; //request to close connection

  //get number of bytes to send
  len = min( pConn->SndDataLen - (unsigned short)Pos, MaxLen );
  if( len == 0 ) //nothing to send
    return 0;

  //no variable in variable buffer
  var = 0;
  VarLen = 0;
  VarPos = 0;
  //if part that should be sent starts with variable
  src = pConn->pSndData + (uint16_t)Pos;
  chr = (char)pgm_read_byte_near( src ); //read first character
  if( pConn->Vars && (unsigned char)chr >= 0x80 )
  {
    //get variable
    var = chr;
    VarLen = HttpGetVariable( var, VarBuffer );
    //get position in variable
    for( VarPos = 0; src > pConn->pSndData; VarPos++, src-- )
      if( (char)pgm_read_byte_near( src ) != var ) //if normal character of other variable, we found the begin of the variable
        { VarPos++; break; }
  }
  //copy data to buffer
  dest = pBuffer;
  for( j = 0; j < len; j++ )
  {
    //read current character
    chr = (char)pgm_read_byte_near( src );
    //variable
    if( pConn->Vars && (unsigned char)chr >= 0x80 )
    {
      //new variable
      if( var != chr )
      {
        //get variable
        var = chr;
        VarLen = HttpGetVariable( var, VarBuffer );
        VarPos = 0;
      }
      //copy next character of variable
      if( VarPos < VarLen ) //get next character of variable
        *dest = VarBuffer[VarPos++];
      else
        *dest = ' '; //fill rest of variable with spaces
    }
    //normal character
    else
    {
      *dest = chr;
      VarPos = 0; //reset variable position (needed if same variable appears two times in a row)
    }
    //next character
    src++;
    dest++;
  } //for( j ...

  //return length of data in buffer
  return len;
}

//called when data was sent and ACKed
void HttpSent( unsigned char ConnNo, unsigned long Pos )
{
  //nothing needs to be done here
}

//called when data was received
//must return new window size (not smaller than curWnd)
unsigned short HttpReceived( unsigned char ConnNo, unsigned long Pos, unsigned char * pBuffer, unsigned short Len, unsigned short curWnd )
{
  unsigned char i;
  struct HttpConnection * pConn;

  //find connection in table
  for( i = 0; i < count( HttpConns ); i++ )
    if( HttpConns[i].ConnNo == ConnNo )
      break;
  if( i >= count( HttpConns ) ) //connection not found
    return max( curWnd, 256 ); //ignore this connection (will be closed in first call of HttpSend)
  //get pointer to connection
  pConn = &HttpConns[i];

  //received duplicate data or missed some data
  //(just to be on the safe side, this should never happen)
  if( pConn->RcvCnt != Pos )
  {
    //close connection
    TcpClose( pConn->ConnNo );
    pConn->ConnNo = 0xFF;
    return max( curWnd, 256 );
  }

  //process received data
  for( ; Len > 0; pBuffer++, Len--, pConn->RcvCnt++ )
  {
    //store character in receive buffer (if it fits into it)
    if( pConn->RcvBufLen < sizeof( pConn->RcvBuf ) )
    {
      pConn->RcvBuf[pConn->RcvBufLen] = *pBuffer;
      pConn->RcvBufLen++;
    }
 
    //actions according to state
    switch( pConn->RcvState )
    {

      //receiving HTTP command (e.g GET)
      case HTTP_RCV_CMD:
        if( *pBuffer == '\r' || *pBuffer == '\n' ) //newline
        {
          pConn->RcvBufLen--; //remove newline from buffer
          HttpParseCommand( pConn ); //parse command
          pConn->PreHttp1 = 1; //older than HTTP 1.0
          pConn->File = HTTP_FILE_INDEX; //send index file
          pConn->RcvState = HTTP_RCV_DONE; //receiving completed
          pConn->RcvBufLen = 0; //empty receive buffer
          HttpProcessRequest( pConn ); //now process request
        }
        else if( *pBuffer == ' ' || *pBuffer == '\t' ) //whitespace
        {
          pConn->RcvBufLen--; //remove whitespace from buffer
          HttpParseCommand( pConn ); //parse command
          pConn->RcvState = HTTP_RCV_FILE; //now receive filename
          pConn->RcvBufLen = 0; //empty receive buffer
        }
        break;
      
      //receiving filename
      case HTTP_RCV_FILE:
        if( *pBuffer == '\r' || *pBuffer == '\n' ) //newline
        {
          pConn->RcvBufLen--; //remove newline from buffer
          HttpParseFile( pConn ); //parse file
          pConn->PreHttp1 = 1; //older than HTTP 1.0
          pConn->RcvState = HTTP_RCV_DONE; //receiving completed
          pConn->RcvBufLen = 0; //empty receive buffer
          HttpProcessRequest( pConn ); //now process request
        }
        else if( *pBuffer == ' ' || *pBuffer == '\t' ) //whitespace
        {
          pConn->RcvBufLen--; //remove whitespace from buffer
          HttpParseFile( pConn ); //parse file
          pConn->PreHttp1 = 0; //HTTP 1.0 or newer
          pConn->RcvState = HTTP_RCV_HDR_END; //now receive header until end
          pConn->RcvBufLen = 0; //empty receive buffer
        }
        break;

      //receiving header until end
      case HTTP_RCV_HDR_END:
        if( *pBuffer != '\r' && *pBuffer != '\n' ) //not a newline
        {
          pConn->RcvBufLen = 0; //empty receive buffer
          break;
        }
        if( (pConn->RcvBufLen == 2 && memcmp( pConn->RcvBuf, "\r\r", 2 ) == 0) || //CR CR
            (pConn->RcvBufLen == 2 && memcmp( pConn->RcvBuf, "\n\n", 2 ) == 0) || //LF LF
            (pConn->RcvBufLen == 3 && memcmp( pConn->RcvBuf, "\r\n\n", 3 ) == 0) || //CR LF LF
            (pConn->RcvBufLen == 3 && memcmp( pConn->RcvBuf, "\n\r\n", 3 ) == 0) || //LF CR LF
            pConn->RcvBufLen == 4 ) //CR LF CR LF (or some other combination of 4 times CR or LF)
        {
          pConn->RcvState = HTTP_RCV_DONE; //receiving completed
          pConn->RcvBufLen = 0; //empty receive buffer
          HttpProcessRequest( pConn ); //now process request
        }
        break;

      //receiving completed
      case HTTP_RCV_DONE:
        pConn->RcvBufLen = 0; //ignore any additional data
        break;

    } //switch( pConn->RcvState )
  } //for( ; Len > 0; pBuffer++, Len--, pConn->RcvCnt++ )

  //return at least 256 bytes window size
  // - we are always able to receive data
  return max( curWnd, 256 );
}

//http notification functions
struct TcpNotify HttpNotify = //(extern)
{
  .Connect = HttpConnect,
  .Close = HttpClose,
  .Send = HttpSend,
  .Sent = HttpSent,
  .Received = HttpReceived,
};

//initialize
void HttpInit( void ) //(extern)
{
  unsigned char i;

  //no HTTP connections yet
  for( i = 0; i < count( HttpConns ); i++ )
    HttpConns[i].ConnNo = 0xFF;
}

