/* 8BitAmEthernet - LuminoWand
 * version 0.2.1 date 2005-08-17
 * Copyright 2005 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 * a project of CCC-AC - http://www.cccac.de/
 */

#include <string.h>
#include <avr/io.h>

#include <lumino.h>
#include <udp.h>

#define SEL 0x04
#define OE 0x02
#define CLK 0x01

static unsigned char LuminoHalfPicBuf[1441];

//initialize
void LuminoInit( void ) //(extern)
{
  //initialize output ports
  DDRD = 0xFF;
  PORTD = 0x00;
  DDRF = 0xFF;
  PORTF = 0x00;
}

//set bits of output
//must be called with interrupts turned off
static void LuminoSet( unsigned char Data, unsigned char Ctrl )
{
  PORTD = Data;
  PORTF = Ctrl;
}

//output a line
//must be called with interrupts disabled
static void LuminoLine( unsigned char Data[30] )
{
  unsigned char i;
  PORTF |= SEL;
  for( i = 0; i < 29; i++ )
  {
    PORTD = Data[i];
    PORTF |= CLK;
    PORTF &= ~CLK;
  }
  PORTF &= ~SEL;
  PORTD = Data[29];
  PORTF |= CLK;
  PORTF &= ~CLK;
}

//output a half picture
// - last byte is sequence number (bit 0 == 0 ---> 1st half)
//must be called with interrupts disabled
static void LuminoHalfPic( unsigned char Data[1441] )
{
  unsigned char i, j, *ptr;

  //first half
  if( (Data[1440] & 0x01) == 0x00 )
  {
    //store 1st half of picturein memory
    memcpy( LuminoHalfPicBuf, Data, 1441 );
    return;
  }

  //2nd half does not match 1st half
  if( (LuminoHalfPicBuf[1440] ^ Data[1440]) != 0x01 )
    //drop 2nd half
    return;

  //turn off display
  PORTF &= ~OE;

  //output 1st half
  ptr = LuminoHalfPicBuf;
  for( j = 0; j < 48; j++ )
  {
    PORTF &= ~SEL;
    PORTD = *(ptr++);
    PORTF |= CLK;
    PORTF &= ~CLK;
    PORTF |= SEL;
    for( i = 0; i < 29; i++ )
    {
      PORTD = *(ptr++);
      PORTF |= CLK;
      PORTF &= ~CLK;
    }
  }

  //output 2nd half
  ptr = Data;
  for( j = 0; j < 48; j++ )
  {
    PORTF &= ~SEL;
    PORTD = *(ptr++);
    PORTF |= CLK;
    PORTF &= ~CLK;
    PORTF |= SEL;
    for( i = 0; i < 29; i++ )
    {
      PORTD = *(ptr++);
      PORTF |= CLK;
      PORTF &= ~CLK;
    }
  }

  //turn on display
  PORTF |= OE;
}

//process received Lumino data
//must be called with interrupts disabled
static void LuminoRecv( unsigned char * pData, unsigned short Length )
{
  //length determines packet type
  switch( Length )
  {
    //set output bits
    case 2:
      LuminoSet( pData[0], pData[1] );
      break;
    //output line
    case 30:
      LuminoLine( pData );
      break;
    //half picture
    case 1441:
      LuminoHalfPic( pData );
      break;
  }
}

//process a received UDP Lumino packet
//must be called with interrupts disabled
void LuminoUdpRecv( unsigned char * pData, unsigned short Length ) //(extern)
{
  struct UdpPacket * pUdpPack;

  //convert pointer to UDP packet
  //(this saves us from always casting pData)
  pUdpPack = (struct UdpPacket *)pData;

  //received data
  LuminoRecv( pData + sizeof( struct UdpPacket ), Length - sizeof( struct UdpPacket ) );
}
